<?php

namespace App\Http\Actions\Pages;

use App\Models\Product;
use App\Models\Category;
use Illuminate\View\View;
use Illuminate\Http\Request;
use App\Cache\CategoryTreeCache;
use Illuminate\Support\Facades\DB;
use App\Services\Models\CategoryService;
use Illuminate\Database\Eloquent\Builder;
use App\Services\Models\CategoryAttributeService;
use Symfony\Component\CssSelector\Exception\InternalErrorException;

class ShopPageAction
{
    public function __invoke(Request $request): View
    {

        $query = Product::query();

        // apply category filter note that when a category i selected all its children will be selected too
        if (isset($request->category)) {
            $children = CategoryService::getChildrenById(cacheable(CategoryTreeCache::class)->get(), $request->category);
            $ids = collect($children)->pluck('id');
            $ids->push($request->category);
            $query->whereIn('category_id', $ids);
        }

        if (isset($request->search)) {
            $query->where('title', 'like', '%' . $request->search . '%');
        }

        if (session()->has('shop.order_by')) {
            $orderBy = session('shop.order_by');

            switch ($orderBy) {
                case 'latest':
                    $query->latest();
                    break;
                case 'oldest':
                    $query->oldest();
                    break;
                case 'best seller':
                    $query->oldest();
                    break;
                case 'cheapest':
                    $query->orderBy('price', 'asc');
                    break;
                case 'most_expensive':
                    $query->orderBy('price', 'desc');
                    break;
                default:
                    break;
            }
        }

        if (session()->has('shop.only_available')) {
            // $query->where('inventory', '>', 0);


            $query->whereHas('inventories', function ($q) {
                $q->select(DB::raw('SUM(inventory)'))
                    ->havingRaw('SUM(inventory) > 0');
            });
        }

        if (session()->has('shop.only_special_offer')) {
            $query->where('off_rate', '>', 0);
        }

        if (session()->has('shop.price.min') && session()->has('shop.price.max')) {
            $query->whereBetween('price', [str_replace(',', '', session('shop.price.min')), str_replace(',', '', session('shop.price.max'))]);
        }
        if (isset($request->category)) {
            $categoryFilterAttributes = CategoryAttributeService::getCategoryAttributes(
                category: $request->category,
                query: true
            )
                ->where('is_filterable', 1)
                ->get()
                ->keyBy('name')
                ->toArray();

            if (session()->has('shop.attributes')) {
                $filters = session('shop.attributes');
                $categoryFilterAttributes = array_intersect_key($categoryFilterAttributes, $filters);

                foreach ($categoryFilterAttributes as $key => $attribute) {
                    $attributeId = $attribute['id'];
                    $filterValue = $filters[$key];
                    if (!$filterValue) {
                        continue;
                    }

                    $filterType = $attribute['filter_type'];
                    switch ($filterType) {
                        case 'text':
                            $query->whereHas(
                                'attributesValues',
                                fn(Builder $query) => $query->where('advertise_type_attribute_id', $attributeId)
                                    ->where('value_type', 'fixed')
                                    ->where('value', 'like', "%{$filterValue}%")
                            );
                            break;

                        case 'range':
                            $query->whereHas(
                                'attributesValues',
                                fn(Builder $query) => $query->where('advertise_type_attribute_id', $attributeId)
                                    ->where('value_type', 'range')
                                    ->where('min_value', '<=', $filterValue['min'] ?? 0)
                                    ->where('max_value', '>=', $filterValue['max'] ?? 0)
                            );
                            break;

                        case 'checkbox':
                            $query->whereHas(
                                'attributesValues',
                                fn(Builder $query) => $query->where('advertise_type_attribute_id', $attributeId)
                                    ->whereIn('value', $filterValue)
                            );
                            break;

                        default:
                            throw new InternalErrorException("Invalid filter type ($filterType).");
                    }
                }
            }
        }


        $products = $query->paginate(perPage: 10);
        return view('pages.shop', compact('products'));
    }
}
