<?php

namespace App\Http\Controllers\Admin\Faq;

use App\Http\Controllers\Controller;
use Artemis\Faq\Enums\QuestionStatusTypeEnum;
use Artemis\Faq\Models\Question;
use Illuminate\Http\Request;

class QuestionController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $questions = Question::all();
        return view('pages.admin.faq.questions.index', compact('questions'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $statuses = QuestionStatusTypeEnum::cases();
        return view('pages.admin.faq.questions.create', compact('statuses'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'user_id' => 'required|integer|exists:users,id',
            'title' => 'required|string|max:255',
            'slug' => 'required|string|unique:faq_questions,slug|max:255',
            'body' => 'required|string',
            'category_id' => 'required|integer|exists:faq_categories,id',
            'status' => 'required|string|in:' . implode(',', array_column(QuestionStatusTypeEnum::cases(), 'value')),
        ]);

        Question::create($validated);

        return redirect()->route('admin.faq.questions.index')
            ->with('success', __('Question created successfully.'));
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $question = Question::findOrFail($id);
        return view('pages.admin.faq.questions.show', compact('question'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $question = Question::findOrFail($id);
        $statuses = QuestionStatusTypeEnum::cases();
        return view('pages.admin.faq.questions.edit', compact('question', 'statuses'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $validated = $request->validate([
            'user_id' => 'required|integer|exists:users,id',
            'title' => 'required|string|max:255',
            'slug' => 'required|string|unique:faq_questions,slug,' . $id . '|max:255',
            'body' => 'required|string',
            'category_id' => 'required|integer|exists:faq_categories,id',
            'status' => 'required|string|in:' . implode(',', array_column(QuestionStatusTypeEnum::cases(), 'value')),
        ]);

        $question = Question::findOrFail($id);
        $question->update($validated);

        return redirect()->route('admin.faq.questions.index')
            ->with('success', __('Question updated successfully.'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $question = Question::findOrFail($id);
        $question->delete();

        return redirect()->route('admin.faq.questions.index')
            ->with('success', __('Question deleted successfully.'));
    }
}
