<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Services\Models\CartService;

class CartController extends Controller
{
    protected CartService $cart;

    public function __construct(CartService $cart)
    {
        $this->cart = $cart;
    }

    public function index()
    {
        $items = $this->cart->items();
        $total = $this->cart->total();
        return view('pages.cart', compact('items', 'total'));
    }

    public function add(Request $request)
    {
        $data = $request->validate([
            'product_id'   => 'required|exists:products,id',
            'inventory_id' => 'required|exists:product_inventories,id',
            'shop_id'      => 'required|exists:shops,id',
            'quantity'     => 'sometimes|integer|min:1',
            'variant'      => 'nullable|array',
            'price'        => 'nullable|string',
            'discount'     => 'nullable|string',
            'coupon'       => 'nullable|string',
            'data'         => 'nullable|array',
        ]);

        $this->cart->add(
            $data['product_id'],
            $data['inventory_id'],
            $data['shop_id'],
            $data['quantity'] ?? 1,
            $data['variant']   ?? [],
            $data['price']     ?? null,
            $data['discount']  ?? null,
            $data['coupon']    ?? null,
            $data['data']      ?? []
        );

        return back()->with('success', 'Added to cart.');
    }

    public function update(Request $request)
    {
        $data = $request->validate([
            'product_id'   => 'required|exists:products,id',
            'inventory_id' => 'nullable|exists:product_inventories,id',
            'shop_id'      => 'nullable|exists:shops,id',
            'quantity'     => 'sometimes|integer|min:1',
            'variant'      => 'nullable|array',
            'price'        => 'nullable|string',
            'discount'     => 'nullable|string',
            'coupon'       => 'nullable|string',
            'data'         => 'nullable|array',
        ]);

        $this->cart->updateQuantity(
            $data['product_id'],
            $data['quantity'] ?? 1,
            $data['inventory_id'],
            $data['shop_id'],
        );
        return redirect()->route('cart.index')->with('success', 'Cart updated.');
    }

    public function remove(Request $request)
    {
        $data = $request->validate([
            'product_id' => 'required|integer|exists:products,id',
        ]);
        $this->cart->remove($data['product_id']);
        return redirect()->route('cart.index')->with('success', 'Item removed from cart.');
    }

    public function clear()
    {
        $this->cart->clear();
        return redirect()->route('cart.index')->with('success', 'Cart cleared.');
    }
}
