<?php

namespace App\Models;

use RuntimeException;
use App\Traits\HasMeta;
use App\Traits\Likeable;
use App\Traits\Taggable;
use Illuminate\Support\Str;
use PhpParser\Node\Expr\FuncCall;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;

class Product extends Model
{
    use HasFactory, SoftDeletes, HasMeta, Likeable ,Taggable;

    protected $fillable = [
        'user_id',
        'category_id',
        'title',
        'code',
        'slug',
        'short_description',
        'content',
        'inventory',
        'weight',
        'length',
        'width',
        'height',
        'price',
        'discount_price',
        'off_rate',
        'second_hand',
        'consumable',
        'status',
        'brand_img',
        'attributes',
    ];

    /**
     * The part below is for this model relationships
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function category(): BelongsTo
    {
        return $this->belongsTo(Category::class);
    }
    public function inventories(): HasMany
    {
        return $this->hasMany(ProductInventory::class, 'product_id');
    }

    public function attribute(): HasMany
    {
        return $this->hasMany(ProductAttributeValue::class, 'product_id');
    }
    public function metas(): HasMany
    {
        return $this->hasMany(ProductMeta::class, 'product_id');
    }

    public function comments()
    {
        return $this->morphMany(Comment::class, 'commentable');
    }

    public function sellers(): HasManyThrough
    {
        return $this->hasManyThrough(Shop::class, ProductInventory::class, 'product_id', 'id', 'id', 'shop_id');
    }
    public function viewers(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'product_views')
            ->withTimestamps();
    }
    public function officeEquipmentGroups()
    {
        return $this->belongsToMany(OfficeEquipmentGroup::class, 'office_equipment_group_product')
            ->withPivot('quantity')
            ->withTimestamps();
    }
    /**
     * The above is for this model relationships
     */

    public static function generateSlug(string $title): string
    {
        $baseSlug = Str::slug($title);
        $slugCount = static::where('slug', 'like', "$baseSlug%")->count();

        return $slugCount === 0 ? $baseSlug : $baseSlug . '-' . ($slugCount + 1);
    }

    public static function generateCode(int $length = 7, int $maxAttempts = 5): string
    {
        $attempts = 0;

        do {
            $code = strtoupper(Str::random($length));
            $attempts++;

            if ($attempts >= $maxAttempts) {
                throw new RuntimeException(__('api.request_failed'), 500);
            }
        } while (static::where('code', $code)->exists());

        return $code;
    }


    /**
     * helper functions
     */
    public function hasSpecialOffer(): bool
    {
        return $this->off_rate > 0;
    }
}
