<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Wallet extends Model
{
    protected $fillable = [
        'user_id',
        'balance',
        'blocked',
        'type',
    ];

    public function increase($amount, $description = null)
    {
        $this->update(['balance' => $this->balance + $amount]);
        $this->transaction()->create([
            'amount' => $amount,
            'type' => 'deposit',
            'description' => $description,
            'balance' => $this->balance,
        ]);
    }

    public function decrease($amount, $description = null)
    {
        $this->update(['balance' => $this->balance - $amount]);
        $this->transaction()->create([
            'amount' => $amount,
            'type' => 'withdraw',
            'description' => $description,
            'balance' => $this->balance,
        ]);
    }

    public function block($amount, $description = null)
    {
        $balance = $this->balance;
        $this->update(['balance' => $balance - $amount, 'block' => $amount]);
        $this->transaction()->create([
            'amount' => $amount,
            'type' => 'block',
            'description' => $description,
            'balance' => $this->balance,
        ]);
    }

    public function unblock($amount, $description = null)
    {
        $balance = $this->balance;
        $blocked = $this->blocked;
        $this->update(['balance' => $balance + $amount, 'block' => $blocked - $amount]);
        $this->transaction()->create([
            'amount' => $amount,
            'type' => 'unblock',
            'description' => $description,
            'balance' => $this->balance,
        ]);
    }

    public function withdrawableAmount()
    {
        $date = \Carbon\Carbon::today()->subDays(3);
        $blocked_amount = $this->transaction()->where('type', 'deposit')->where('created_at', '>=', $date)->sum('amount');

        return (($this->balance - $blocked_amount) < 0) ? 0 : $this->balance - $blocked_amount;
    }

    /**
     * The part below is for this model relationships
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function transaction()
    {
        return $this->hasMany(WalletTransaction::class);
    }

    public function payment()
    {
        return $this->morphMany(Payment::class, 'payable');
    }

    public function settlements()
    {
        return $this->hasMany(WalletSettlement::class);
    }
    /**
     * The above is for this model relationships
     */
}
