<?php

namespace App\Services\Models;

use App\Models\Cart;
use App\Models\CartItem;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Auth;

class CartService
{
    /**
     * Retrieve or create the active cart for this session/user
     */
    public function getCart(): Cart
    {
        if (Auth::check()) {
            // fetch or make an “active” cart for the user
            return Cart::firstOrCreate(
                ['user_id' => Auth::id(), 'status' => 'active'],
                ['session_id' => null]
            );
        }

        // guest: find by session_id, or create one
        $sid = Session::get('cart_session_id');
        if (! $sid) {
            $sid = Str::uuid();
            Session::put('cart_session_id', $sid);
        }

        return Cart::firstOrCreate(
            ['session_id' => $sid, 'status' => 'active'],
            ['user_id' => null]
        );
    }

    public function updatedCartAddress($addressId)
    {
        $cart = $this->getCart();
        $cart->address_id = $addressId;
        $cart->save();
        return $this->getCart();
    }
    public function add(
        int   $productId,
        ?int   $inventoryId = null,
        ?int   $shopId = null,
        int   $quantity    = 1,
        ?array $variant     = [],
        ?string $price     = null,
        ?string $discount  = null,
        ?string $coupon    = null,
        ?array $data        = []
    ): void {
        $cart = $this->getCart();

        $item = CartItem::firstOrNew([
            'cart_id'      => $cart->id,
            'product_id'   => $productId,
            'inventory_id' => $inventoryId,
            'shop_id'      => $shopId,
            // if you want variant‐distinction: uncomment next line
        ]);

        $item->quantity  = ($item->quantity ?? 0) + $quantity;
        $item->variant   = is_array($variant) ? json_encode($variant) : $variant;
        $item->price     = $price;
        $item->discount  = $discount;
        $item->coupon    = $coupon;
        $item->data      = is_array($data) ? json_encode($variant) : $variant;
        $item->save();
    }

    public function updateCoupon(string $coupon,): void
    {
        $cart = $this->getCart();
        $cart->coupon = $coupon;
        $cart->save();
    }

    public function updateQuantity(
        int $productId,
        int $quantity,
        ?int   $inventoryId,
        ?int   $shopId,
    ): void {
        $cart = $this->getCart();

        if ($quantity <= 0) {
            CartItem::where([
                'cart_id'    => $cart->id,
                'product_id' => $productId,
            ])
                ->when(isset($inventoryId), function ($query) use ($inventoryId) {
                    $query->where('inventory_id', $inventoryId);
                })
                ->when(isset($shopId), function ($query) use ($shopId) {
                    $query->where('shop_id', $shopId);
                })->delete();
            return;
        }

        $item = CartItem::where([
            'cart_id'    => $cart->id,
            'product_id' => $productId,
        ])
            ->when(isset($inventoryId), function ($query) use ($inventoryId) {
                $query->where('inventory_id', $inventoryId);
            })
            ->when(isset($shopId), function ($query) use ($shopId) {
                $query->where('shop_id', $shopId);
            })
            ->first();

        if ($item) {
            $item->quantity = $quantity;
            $item->save();
        }
    }

    public function update(
        int   $productId,
        ?int   $inventoryId,
        ?int   $shopId,
        int   $quantity,
        array $variant     = [],
        ?string $price     = null,
        ?string $discount  = null,
        ?string $coupon    = null,
        array $data        = []
    ): void {
        $cart = $this->getCart();

        if ($quantity <= 0) {
            CartItem::where([
                'cart_id'      => $cart->id,
                'product_id'   => $productId,
                'inventory_id' => $inventoryId,
                'shop_id'      => $shopId,
            ])->delete();
            return;
        }

        CartItem::updateOrCreate(
            [
                'cart_id'      => $cart->id,
                'product_id'   => $productId,
                'inventory_id' => $inventoryId,
                'shop_id'      => $shopId,
                // 'variant'      => json_encode($variant),
            ],
            [
                'quantity'  => $quantity,
                'variant'   => $variant,
                'price'     => $price,
                'discount'  => $discount,
                'coupon'    => $coupon,
                'data'      => $data,
            ]
        );
    }


    public function remove(int $productId): void
    {
        $cart = $this->getCart();
        CartItem::where('cart_id', $cart->id)
            ->where('product_id', $productId)
            ->delete();
    }

    public function clear(): void
    {
        $cart = $this->getCart();
        $cart->items()->delete();
    }

    public function items()
    {
        return $this->getCart()
            ->items()
            ->with('product')
            ->get();
    }

    public function count(): int
    {
        return $this->items()->sum('quantity');
    }

    public function total(): float
    {
        return $this->items()->reduce(function ($sum, $item) {
            return $sum + (($item->price + $item->discount) * $item->quantity);
        }, 0);
    }
    public function totalPayable(): float
    {
        return $this->items()->reduce(function ($sum, $item) {
            return $sum + ($item->price * $item->quantity);
        }, 0);
    }
}
