<?php

namespace App\Services\Models;

use App\Models\Product;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\DB;
use App\Services\FileUploadService;
use Illuminate\Support\Facades\Validator;

class ProductService
{
    public static function createOrUpdate($data, $product = null)
    {
        
        return DB::transaction(function () use ($data, $product) {
            // Handle brand image
            if (isset($data['image'])) {
                $brand_image =  (new FileUploadService($data['image']))->upload();
            }

            $productData = self::prepareData($data);
            $productData['brand_img'] = isset($brand_image) ? $brand_image['full_url'] : null;

            // Create product or update existing product
            if (isset($product)) {
                $product->update(removeNullValues($productData));
                $product->refresh();
            } else {
                $product = Product::create($productData);
            }


            if (isset($data['images'])) {
                // Store gallery images
                foreach ($data['images'] as $image) {
                    if ($image instanceof UploadedFile) {
                        $image = (new FileUploadService($image))->upload();
                        $product->metas()->create([
                            'key' => 'gallery',
                            'value' => $image['full_url'],
                            'data' => json_encode($image),
                        ]);
                    }
                }
            }
            if (isset($data['tag'])) {
                $product->tags()->sync($data['tag']);
            }

            if (isset($data['delete_images'])) {
                $product->metas()->whereIn('id', $data['delete_images'])->delete();
            }

            // Store attributes
            $result = self::storeProductAttributes($product, $data['attributes'] ?? []);

            if ($result !== true) {
                return $result;
            }

            return $product;
        });
    }

    private static function storeProductAttributes($product, $attributes): array|bool
    {
        // find attributes
        $attributeQuery = CategoryAttributeService::getCategoryAttributes($product->category, true);

        $categoryAttributes = $attributeQuery->where('is_price_effective', 0)->get();

        // validate attributes
        $rules = $categoryAttributes->mapWithKeys(function ($attribute) {
            return [
                $attribute->name => $attribute->is_required ? 'required' : 'nullable',
            ];
        })->toArray();


        $validator = Validator::make(array_map(fn($item) => $item['value'], $attributes), $rules);

        if ($validator->fails()) {
            return [
                'status' => false,
                'errors' => $validator->errors(),
            ];
        }

        // store attributes
        $result = ProductAttributeService::createOrUpdateProductAttribute($product, $attributes);

        return true;
    }

    private static function prepareData($data)
    {
        // Calculate discount_price and off_rate if one of them is missing
        if (isset($data['price'])) {
            if (isset($data['off_rate']) && !isset($data['discount_price'])) {
                $data['discount_price'] = $data['price'] - ($data['price'] * $data['off_rate'] / 100);
            } elseif (isset($data['discount_price']) && !isset($data['off_rate'])) {
                $data['off_rate'] = (($data['price'] - $data['discount_price']) / $data['price']) * 100;
            }
        }


        return [
            'user_id' => $data['user_id'],
            'category_id' => $data['category_id'],
            'title' => $data['title'],
            'code' => $data['code'] ?? Product::generateCode(),
            'slug' => isset($data['slug']) ? Product::generateSlug($data['slug']) : Product::generateSlug($data['title']),
            'short_description' => $data['short_description'],
            'content' => $data['content'],
            'inventory' => $data['inventory'],
            'weight' => $data['weight'],
            'length' => $data['length'],
            'width' => $data['width'],
            'height' => $data['height'],
            'price' => $data['price'],
            'discount_price' => $data['discount_price'] ?? null,
            'off_rate' => $data['off_rate'] ?? null,
            'second_hand' => $data['second_hand'] ? 1 : 0,
            'consumable' => $data['consumable'] ? 1 : 0,
            'status' => $data['status'],
            'brand_img' => null,
            'attributes' => isset($data['attributes']) ? json_encode($data['attributes']) : json_encode([]),
        ];
    }

    public function getSellers($product)
    {
        $sellers = $product->sellers;

        return $sellers;
    }
}
