<?php

namespace Artemis\Faq\Http\Controllers\api\v1;

use Str;
use Illuminate\Http\Request;
use Artemis\Faq\Models\Question;
use Illuminate\Http\JsonResponse;
use Artemis\Faq\Events\ResourceViewedEvent;
use Artemis\Faq\Enums\QuestionStatusTypeEnum;
use Artemis\Faq\Http\Resources\QuestionResource;
use Artemis\Faq\Http\Controllers\api\ApiController;
use Artemis\Faq\Http\Requests\QuestionStoreRequest;
use Artemis\Faq\Http\Requests\QuestionUpdateRequest;
use Symfony\Component\HttpFoundation\Response as ResponseCode;

class QuestionController extends ApiController
{
    public function index(Request $request): JsonResponse
    {
        $perPage = $this->getperPage($request);
        $page = $this->getPage($request);
        
        $query = Question::query();
        $query->when($request->user_id, fn($query) => $query->where('user_id', $request->user_id))
            ->when($request->category_id, fn($query) => $query->where('category_id', $request->category_id))
            ->when($request->status, fn($query) => $query->where('status', $request->status));
        
        $questions = $query->paginate(perPage: $perPage, page: $page);
        $data = [
                'questions' => QuestionResource::collection($questions),
            ] + $this->paginateResult($questions);
        
        return $this->success($data);
    }
    
    public function store(QuestionStoreRequest $request): JsonResponse
    {
        Question::create([
            'user_id' => $request->user()->id,
            'title' => $request->title,
            'body' => $request->body,
            'slug' => $request->slug ? Str::slug($request->slug) : generateSlugForModel(
                $request->title,
                Question::class
            ),
            'category_id' => $request->category_id,
            'status' => QuestionStatusTypeEnum::OPEN,
        ]);
        
        return $this->success(status: ResponseCode::HTTP_CREATED);
    }
    
    public function show(Question $question): JsonResponse
    {
        ResourceViewedEvent::dispatch($question);
        
        return $this->success(new QuestionResource($question));
    }
    
    public function update(QuestionUpdateRequest $request, Question $question): JsonResponse
    {
        $question->update($request->validated());
        
        return $this->success();
    }
    
    public function destroy(Question $question): JsonResponse
    {
        $question->delete();
        
        return $this->success();
    }
}
