<?php

namespace App\Http\Controllers\Admin;

use App\Models\Product;
use App\Models\Category;
use App\Services\SweetAlert;
use Illuminate\Http\Request;
use App\Enums\ProductStatusEnum;
use App\Http\Controllers\Controller;
use App\Services\Models\ProductService;
use App\Http\Requests\Admin\ProductRequest;
use Illuminate\Support\Facades\Route;

class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $query = Product::query();

        if (request()->has('search')) {
            $query->where('name', 'like', '%' . request('search') . '%');
        }
        if (request()->has('category_id')) {
            $query->where('category_id', request('category_id'));
        }

        if (Route::is('admin.shop.product-selection')) {
            $query->where('status', ProductStatusEnum::Published);
        }

        $products = $query->paginate(20);

        return view('pages.admin.product.list', compact('products'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        if (isset($request->category_id) && Category::where('id', $request->category_id)->exists()) {
            $category = Category::find($request->category_id);
        } else {
            return redirect()->route('admin.categories.index')->with('sweetalert', SweetAlert::error(__('error'), __('before creating a product you need to select a category')));
        }

        return view('pages.admin.product.create', compact('category'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(ProductRequest $request)
    {
        $product = (new ProductService())->createOrUpdate($request->validated());

        return redirect()->route('admin.product.index')->with('sweetalert', SweetAlert::success(__('success'), __('product created successfully')));
    }

    /**
     * Display the specified resource.
     */
    public function show(Product $product)
    {
        return view('pages.admin.product.show', compact('product'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Request $request, Product $product)
    {
        if (isset($product->category_id) && Category::where('id', $product->category_id)->exists()) {
            $category = Category::find(id: $product->category_id);
        } else {
            return redirect()->route('admin.product.show', $product->id)->with('sweetalert', SweetAlert::error(__('error'), __('there was problem with the edit')));
        }

        // dd($product->attribute->keyBy('category_attribute_id'));
        return view('pages.admin.product.edit', compact('category', 'product'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(ProductRequest $request, Product $product)
    {
        $result = (new ProductService())->createOrUpdate($request->validated(), $product);

        return redirect()->route('admin.product.show', $product->id)->with('sweetalert', SweetAlert::success(__('success'), __('product created successfully')));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Category $category, Product $product)
    {
        $product->delete();

        return redirect()->route('admin.product.index')->with('sweetalert', SweetAlert::success(__('success'), __('product deleted successfully')));
    }
}
