<?php

namespace App\Http\Controllers\Admin\Shop;

use App\Services\SweetAlert;
use Illuminate\Http\Request;
use App\Models\ProductInventory;
use App\Http\Controllers\Controller;
use App\Enums\ProductInventoryStatusEnum;

class InventoryController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $query = ProductInventory::query();

        if (request()->has('search')) {
            $query->where('product_id', 'like', '%' . request('search') . '%');
        }

        if (request()->has('status')) {
            $query->where('status', request('status'));
        }

        $query->where('shop_id', auth()->user()->id);

        $inventories = $query->paginate(20);

        return view('pages.admin.shop.inventory.index', compact('inventories'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer|min:1',
            'price' => 'required|integer',
        ]);

        ProductInventory::updateOrCreate(
            [
                'product_id' => $request->product_id,
                'user_id' => auth()->user()->id,
                'shop_id' => auth()->user()->shop->id,
            ],
            [
                'status' => ProductInventoryStatusEnum::Available,
                'price' => $request->price,
                'inventory' => $request->quantity,
            ],
        );
        return redirect()->route('admin.shop.inventory.index')->with('sweetalert', SweetAlert::success(__('success'), __('inventory created successfully')));
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $inventory = ProductInventory::findOrFail($id);

        if (auth()->user()->isAdmin() || $inventory->shop_id !== auth()->user()->id) {
            abort(403, __('You are not authorized to delete this inventory.'));
        }

        $inventory->delete();

        return redirect()->route('admin.shop.inventory.index')->with('sweetalert', SweetAlert::success(__('success'), __('inventory deleted successfully')));
    }
}
