<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Shop;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage; // Added for file handling

class ShopController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $shops = Shop::latest()->paginate(10);
        return view('pages.admin.shop-profile.index', compact('shops'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('pages.admin.shop-profile.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:shops,name',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'description' => 'nullable|string',
            'phone' => 'required|string|max:20',
            'email' => 'required|email|max:255|unique:shops,email',
            'address' => 'required|string|max:255',
            'city' => 'required|string|max:100',
            'state' => 'nullable|string|max:100',
            'postal_code' => 'required|string|max:20',
            'country' => 'required|string|max:100',
        ]);

        $data = $request->except('logo');
        $data['user_id'] = Auth::id(); // Assuming the shop is created by the logged-in admin/user

        if ($request->hasFile('logo')) {
            $data['logo'] = $request->file('logo')->store('shop_logos', 'public_upload');
        }

        $shop = Shop::create($data);

        return redirect()->route('admin.shops.index')->with('success', 'Shop created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Shop $shop)
    {
        return view('pages.admin.shop-profile.show', compact('shop'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Shop $shop)
    {
        return view('pages.admin.shop-profile.edit', compact('shop'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Shop $shop)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:shops,name,' . $shop->id,
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'description' => 'nullable|string',
            'phone' => 'required|string|max:20',
            'email' => 'required|email|max:255|unique:shops,email,' . $shop->id,
            'address' => 'required|string|max:255',
            'city' => 'required|string|max:100',
            'state' => 'nullable|string|max:100',
            'postal_code' => 'required|string|max:20',
            'country' => 'required|string|max:100',
        ]);

        $data = $request->except('logo');

        if ($request->hasFile('logo')) {
            // Delete old logo if exists
            if ($shop->logo) {
                Storage::disk('public_upload')->delete($shop->logo);
            }
            $data['logo'] = $request->file('logo')->store('shop_logos', 'public_upload');
        }

        $shop->update($data);

        return redirect()->route('admin.shops.index')->with('success', 'Shop updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Shop $shop)
    {
        if ($shop->logo) {
            Storage::disk('public')->delete($shop->logo);
        }
        $shop->delete();
        return redirect()->route('admin.shops.index')->with('success', 'Shop deleted successfully.');
    }

    // Optional: Method to approve a shop
    public function approve(Shop $shop)
    {
        $shop->update([
            'is_approved' => true,
            'approved_at' => now(),
        ]);

        $shop->user->assignRole('Shop'); // Assign a role to the user if needed

        return redirect()->route('admin.shops.index')->with('success', 'Shop approved successfully.');
    }

    // Optional: Method to unapprove a shop
    public function unapprove(Shop $shop)
    {
        $shop->update([
            'is_approved' => false,
            'approved_at' => null,
        ]);
        $shop->user->removeRole('Shop'); // Assign a role to the user if needed

        return redirect()->route('admin.shops.index')->with('success', 'Shop unapproved successfully.');
    }
}
