<?php

namespace App\Livewire\Blesser;

use App\Events\UserVisitedProduct;
use App\Models\Product;
use Livewire\Component;
use Livewire\Attributes\On;
use App\Services\Models\CartService;
use App\Services\Models\ProductService;
use App\Services\Models\CategoryAttributeService;

class ProductDetails extends Component
{
    protected CartService $cart;
    public bool $eventDispatched = false;

    public $product;
    public $inventory;
    public $gallery;
    public $sellers;
    public $productVariantInputs;

    public $selectedSeller = null;

    public $price;
    public $discountAmount = 0;

    public function __construct()
    {
        $this->cart = app(CartService::class);
    }

    public function mount($slug)
    {


        $this->product = Product::where('slug', $slug)->firstOrFail();
        $this->inventory = $this->product->inventories->keyBy('shop_id');
        $this->gallery = $this->product->getMetaByKey('gallery');
        $this->sellers = (new ProductService)->getSellers($this->product)->keyBy('id');
        $this->price = $this->product->discount_price ?? $this->product->price;
        $this->discountAmount = $this->product->discount_price != 0 ? $this->product->price - $this->product->discount_price : 0;

        $this->productVariantInputs = CategoryAttributeService::getCategoryPriceEffectiveAttributes($this->product->category);

        if (!$this->eventDispatched) {
            event(new UserVisitedProduct($this->product));
            $this->eventDispatched = true;
        }
    }

    public function selectSeller($sellerId)
    {
        $this->selectedSeller = $sellerId;
        $this->updatePrice();
    }

    #[On('seller-selected')]
    public function updatePrice()
    {

        if ($this->selectedSeller == null) {
            return;
        }

        $inventory = $this->inventory[$this->selectedSeller];
        if ($inventory) {
            $this->price = $inventory->price;
            $this->discountAmount = 0;
        }
    }

    #[On('seller-selected')]
    public function updateVariant() {}

    public function addToCart()
    {
        $this->cart->add(
            $this->product->id,
            $this->inventory[$this->selectedSeller]->id ?? null,
            $this->selectedSeller,
            1,
            [],
            $this->price,
            $this->discountAmount,
            null,
            []
        );

        return redirect()->route('cart.index')->with('success', __('Added to cart.'));
    }
    public function render()
    {

        return view('livewire.blesser.product-details')->extends('layouts.blesser.app')->section('content');
    }
}
