<?php

namespace App\Services;

use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class FileUploadService
{
    protected UploadedFile $file;
    protected string $disk;
    protected string $path;
    protected ?string $customName = null;
    protected array $fileInfo = [];

    public function __construct(UploadedFile $file, string $path = '', string $disk = 'public_upload')
    {
        $this->file = $file;
        $this->path = $path;
        $this->disk = $disk;

        $this->prepareFileInfo();
    }

    public function setDisk(string $disk): static
    {
        $this->disk = $disk;
        return $this;
    }

    public function setCustomName(string $name): static
    {
        $this->customName = $name;
        return $this;
    }

    /**
     * Uploads the file to the specified disk and path.
     *
     * Generates a unique name for the file if a custom name is not provided.
     * Stores the file using the storeAs method.
     * Populates the fileInfo array with relevant file information.
     *
     * @return array An array containing information about the uploaded file, including:
     *               - original_name: The original name of the file.
     *               - extension: The file extension.
     *               - mime_type: The MIME type of the file.
     *               - size: The file size in bytes.
     *               - stored_path: The path where the file is stored.
     *               - stored_name: The final name of the stored file.
     *               - disk: The disk used for storage.
     *               - full_url: The full URL of the stored file.
     *               - visibility: The visibility of the stored file.
     */
    public function upload(): array
    {
        $finalName = $this->customName
            ? $this->customName . '.' . $this->file->getClientOriginalExtension()
            : $this->generateUniqueName();

        $storedPath = $this->file->storeAs($this->path, $finalName, $this->disk);

        $this->fileInfo['original_name'] = $this->file->getClientOriginalName();
        $this->fileInfo['extension']     = $this->file->getClientOriginalExtension();
        $this->fileInfo['mime_type']     = $this->file->getClientMimeType();
        $this->fileInfo['size']          = $this->file->getSize();
        $this->fileInfo['stored_path']   = $storedPath;
        $this->fileInfo['stored_name']   = $finalName;
        $this->fileInfo['disk']          = $this->disk;
        $this->fileInfo['full_url']      = Storage::disk($this->disk)->url($storedPath);
        $this->fileInfo['visibility']    = Storage::disk($this->disk)->getVisibility($storedPath);

        return $this->fileInfo;
    }

    public function getFileInfo(): array
    {
        return $this->fileInfo;
    }

    protected function generateUniqueName(): string
    {
        $extension = $this->file->getClientOriginalExtension();
        $basename = pathinfo($this->file->getClientOriginalName(), PATHINFO_FILENAME);
        $slugified = Str::slug($basename);
        return $slugified . '-' . uniqid() . '.' . $extension;
    }

    protected function prepareFileInfo(): void
    {
        $this->fileInfo = [
            'original_name' => $this->file->getClientOriginalName(),
            'mime_type'     => $this->file->getClientMimeType(),
            'size'          => $this->file->getSize(),
            'extension'     => $this->file->getClientOriginalExtension(),
            'client_path'   => $this->file->getPathname(),
            'stored_path'   => null,
            'stored_name'   => null,
            'disk'          => $this->disk,
            'full_url'      => null,
            'visibility'    => null,
        ];
    }
}
